# Dijkstra's algorithm for shortest paths
# David Eppstein, UC Irvine, 4 April 2002
# Jan Horacek, 30 April 2015

"""
Tento soubor implementuje upraveny Dijstruv algoritmus pro reseni ulohy KSI
"V Hotelu".
Uprava zahrnuje to, ze Dijkstra nehleda cestu grafem mezi 2-ma body, ale mezi
jednim bodem a nejblizsi rozsvicenou mistnosti.

Upraveny algoritmus vychazi z Dijkstry, jehoz implemetaci jsem stahl na nize
zminenem odkazu.
"""

# http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/117228
from priodict import priorityDictionary

def Dijkstra(G, room):
	"""
	Nalezne cestu z room do nejblizsiho rozsviceneho pokoje.

	Puvodni komentare autora, kdy funkce prebirala argumenty Dijkstra(G, start, end): a vracela pouze D,P

		Find shortest paths from the room vertex to all vertices nearer than or equal to the end.

		The input graph G is assumed to have the following representation:
		A vertex can be any object that can be used as an index into a dictionary.
		G is a dictionary, indexed by vertices.  For any vertex v, G[v] is itself a dictionary,
		indexed by the neighbors of v.  For any edge v->w, G[v][w] is the length of the edge.
		This is related to the representation in <http://www.python.org/doc/essays/graphs.html>
		where Guido van Rossum suggests representing graphs as dictionaries mapping vertices
		to lists of outgoing edges, however dictionaries of edges have many advantages over lists:
		they can store extra information (here, the lengths), they support fast existence tests,
		and they allow easy modification of the graph structure by edge insertion and removal.
		Such modifications are not needed here but are important in many other graph algorithms.
		Since dictionaries obey iterator protocol, a graph represented as described here could
		be handed without modification to an algorithm expecting Guido's graph representation.

		Of course, G and G[v] need not be actual Python dict objects, they can be any other
		type of object that obeys dict protocol, for instance one could use a wrapper in which vertices
		are URLs of web pages and a call to G[v] loads the web page and finds its outgoing links.
	
		The output is a pair (D,P) where D[v] is the distance from start to v and P[v] is the
		predecessor of v along the shortest path from s to v.
	
		Dijkstra's algorithm is only guaranteed to work correctly when all edge lengths are positive.
		This code does not verify this property for all edges (only the edges examined until the end
		vertex is reached), but will correctly compute shortest paths even for some graphs with negative
		edges, and will raise an exception if it discovers that a negative edge has caused it to make a mistake.

	Upraveny algoritmus navic vraci jeste onen nejblizsi rozsviceny vrchol,
	celkem tedy: return (v,D,P)
	"""

	D = {}	# dictionary of final distances
	P = {}	# dictionary of predecessors
	Q = priorityDictionary()	# estimated distances of non-final vertices
	Q[room] = 0
	
	for v in Q:
		D[v] = Q[v]
		if (G[v][0]): break         # Tady je jedina uprava -- uprava koncove podminky -- algoritmus konci, kdyz narazi na rozsviceny pokoj
		
		for w in G[v][1]:
			vwLength = D[v] + G[v][1][w]
			if w in D:
				if vwLength < D[w]:
					raise ValueError, "Dijkstra: found better path to already-final vertex"
			elif w not in Q or vwLength < Q[w]:
				Q[w] = vwLength
				P[w] = v
	
	return (v,D,P)
			
def shortestPath(G, room):
	"""
	Nalezne jedinou nejkratsi cestu z pokoje \room do nejblizsiho rozsviceneho pokoje.
	Tato funkce pouze vola Dijkstra a reprezentuje jeho vystup.
	Vystupem teto funkce je seznam, ktery obsahuje vsechny vrcholy v ceste od
	onoho nejblizsiho rozsviceneho pokoje az po pokoj \room.
	"""

	start,D,P = Dijkstra(G, room)
	Path = []

	while True:
		if not G[start][0]: Path.append(start)
		if room == start: break
		start = P[start]
	return Path

"""
Reprezentace hotelu:
   Jedna se o slovnik, kde klicem je index vrcholu.
   Kazdy takovy vrchol si pak pamatuje, jestli je v nem rozsvicene svetlo (false/true),
   a seznam sousedu vcetne "vzdalenosti k nim". Vzdalenost je v nasem pripade
   pocet svetel v CILOVE mistnosti.

Priklad hotelu se tremi mistnostmi:
G = {0: [false, {1:10, 2:8}], - pokoj 0 je zhasnuty a ma sousedy 1, ve kterem je 10 svetel, a 2, ve kterem je 8 svetel
	1: [false, {0:5}],
	2: [true, {0:5}]}
"""

