# encoding: utf-8

"""
KSI 2014, sada 3, priklad 1
Pocitani objemu 2D/3D kopce metodou Monte Carlo.

Uzitecne funkce a konstanty (importovane nize):
- uniform(a, b) vraci nahodne cislo (float) z intervalu [a, b],
- cos(x) pocita kosinus x (x v radianech)
- sqrt(x) pocita odmocninu cisla x
- pi je znama konstanta (3.141592..)
"""

# Verim, ze reseni ulohy je tak primocare, ze nevyzaduje dalsi popis v PDF.
# Jan Horacek, 27.1.2015
# jan.horacek@seznam.cz

from random import uniform
from math import cos, pi, sqrt


# ---------------------------------------------------------------------------
# 2D verze
# ---------------------------------------------------------------------------

def hill_volume_2D(radius, height, attempts=10000):
    """Spocita objem 2D kopce (tj. obsah plochy pod kosinovou krivkou).

    @param radius: polomer kopce (polovina sirky kosinove krivky)
    @param height: vyska kopce (vyska kosinove krivky)
    @param attempts: pocet nahodnych pokusu v ramci Monte Carlo metody (> 0)

    @return: objem 2D kopce [float]
    """

    # Zakladni myslenka:
    # Nahodne strilime souradnice X a Y a divame se, jestli patri pod krivku.
    # Souradnice strilime do obdelniku o sirce 2*radius a vysce height
    # -> podil uspesnych pokusu je pro ziskani absolutni plochy nutne
    #   vynasobit prave obsahem "obdelniku opsaneho".

    # Jak plyne z implementace nize, casova slozitost reseni je O(attempts),
    # kde attempts je pocet pokusu metody Monte Carlo. To plyne jak z principu
    # metody samotne, tak ze zdrojaku nize, ve kterem je jednoduse jen jeden
    # for cyklus.
    # Casovou slozitost matematickych operaci povazuji na konstantni.

    sum = 0

    for i in range(0, attempts):
      x = uniform(-radius, radius)
      y = uniform(0, height)
      sum += (y < height * cos((pi*x)/(2*radius)))

    return (float(sum) / attempts) * (2*radius*height)


# ---------------------------------------------------------------------------
# 3D verze
# ---------------------------------------------------------------------------

def hill_volume_3D(radius, height, attempts=10000):
    """Spocita objem 3D kopce (objem telesa pod orotovanou kosinovou krivkou)

    @param radius: polomer kopce (polovina sirky kosinove krivky)
    @param height: vyska kopce (vyska kosinove krivky)
    @param attempts: pocet nahodnych pokusu v ramci Monte Carlo metody (> 0)

    @return: objem 3D kopce [float]
    """

    # Zakladni myslenka:
    # Nahodne strilime souradnice X, Y a Z a divame se, jestli patri pod krivku.
    # Souradnice strilime do kvadru o sirce 2*radius, hloubce 2*radius a vysce height
    # -> podil uspesnych pokusu je pro ziskani absolutniho objemu kopce nutne
    #   vynasobit prave objemem "kvadru opsaneho" (tj. 4*radius^2*height).

    # Jak plyne z implementace nize, casova slozitost reseni je O(attempts),
    # kde attempts je pocet pokusu metody Monte Carlo. To plyne jak z principu
    # metody samotne, tak ze zdrojaku nize, ve kterem je jednoduse jen jeden
    # for cyklus.
    # Casovou slozitost matematickych operaci povazuji na konstantni.

    sum = 0

    for i in range(0, attempts):
      x = uniform(-radius, radius)
      y = uniform(-radius, radius)
      z = uniform(0, height)
      sum += (z < height * cos((pi*sqrt(x*x+y*y))/(2*radius)))

    return (float(sum) / attempts) * (4*radius*radius*height)
